<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

include(__DIR__ . '/includes/functions.php');
$static_url = dirname($_SERVER['SCRIPT_NAME']) . '/assets/';
include 'session_check.php';

// Função para atualizar apenas os dois campos no config.json
function updateConfigFile($status, $expirationDays) {
    $configFile = __DIR__ . '/api/config.json';
    
    // Ler configurações existentes
    $config = [];
    if (file_exists($configFile)) {
        $config = json_decode(file_get_contents($configFile), true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("Erro ao ler arquivo de configuração");
        }
    }
    
    // Atualizar apenas os campos específicos
    $config['default_status'] = $status;
    $config['default_expiration_days'] = (int)$expirationDays;
    
    // Salvar de volta no arquivo
    if (file_put_contents($configFile, json_encode($config, JSON_PRETTY_PRINT)) === false) {
        throw new Exception("Não foi possível salvar as configurações");
    }
    
    return true;
}

// Carregar configurações atuais
try {
    $configFile = __DIR__ . '/api/config.json';
    $currentConfig = [
        'default_status' => 'inactive',
        'default_expiration_days' => 7
    ];
    
    if (file_exists($configFile)) {
        $fileConfig = json_decode(file_get_contents($configFile), true);
        if ($fileConfig) {
            $currentConfig = array_merge($currentConfig, $fileConfig);
        }
    }
} catch (Exception $e) {
    $error_message = "Erro ao carregar configurações: " . $e->getMessage();
}

// Processar formulário
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $newStatus = $_POST['default_status'] ?? 'inactive';
        $newExpirationDays = (int)($_POST['default_expiration_days'] ?? 7);
        
        // Validações
        if (!in_array($newStatus, ['active', 'inactive'])) {
            throw new Exception("Status inválido");
        }
        
        if ($newExpirationDays <= 0 || $newExpirationDays > 365) {
            throw new Exception("Dias de expiração devem ser entre 1 e 365");
        }
        
        // Atualizar arquivo de configuração
        updateConfigFile($newStatus, $newExpirationDays);
        
        $success_message = "✅ Configurações atualizadas com sucesso!";
        $currentConfig['default_status'] = $newStatus;
        $currentConfig['default_expiration_days'] = $newExpirationDays;
    } catch (Exception $e) {
        $error_message = "⚠️ Erro: " . $e->getMessage();
    }
}

$page_title = "⚙️ Configurações do Sistema";

$page_content = '<div class="container mt-5">
    <div class="card mb-5 mb-xl-12">
        <div class="card-body py-12">
            <div class="d-flex align-items-center mb-7">
                <span class="symbol symbol-50px me-5">
                    <span class="symbol-label bg-light-primary">
                        <i class="ki-outline ki-settings fs-2x text-primary"></i>
                    </span>
                </span>
                <div>
                    <h2 class="mb-1">⚙️ Configurações do Sistema</h2>
                    <p class="text-gray-600">Gerencie as configurações básicas do sistema</p>
                </div>
            </div>';

if (!empty($success_message)) {
    $page_content .= '<div class="alert alert-success d-flex align-items-center" role="alert">
        <i class="ki-outline ki-check-circle fs-2x text-success me-4"></i>
        <div class="d-flex flex-column">
            <h4 class="mb-1 text-success">Sucesso!</h4>
            <span>' . htmlspecialchars($success_message) . '</span>
        </div>
    </div>';
}

if (!empty($error_message)) {
    $page_content .= '<div class="alert alert-danger d-flex align-items-center" role="alert">
        <i class="ki-outline ki-cross-circle fs-2x text-danger me-4"></i>
        <div class="d-flex flex-column">
            <h4 class="mb-1 text-danger">Erro!</h4>
            <span>' . htmlspecialchars($error_message) . '</span>
        </div>
    </div>';
}

$page_content .= '<form method="POST" action="">
    <div class="row g-6 mb-10">
        <div class="col-xl-12">
            <div class="card card-flush">
                <div class="card-header">
                    <h3 class="card-title">🔧 Configurações Principais</h3>
                </div>
                <div class="card-body pt-1">
                    <div class="mb-5">
                        <label for="default_status" class="fs-6 fw-semibold mb-2">🔘 Status Padrão</label>
                        <select class="form-select" id="default_status" name="default_status" required>
                            <option value="inactive" ' . ($currentConfig['default_status'] === 'inactive' ? 'selected' : '') . '>Inactive</option>
                            <option value="active" ' . ($currentConfig['default_status'] === 'active' ? 'selected' : '') . '>Active</option>
                        </select>
                        <div class="form-text">Status padrão para novas ativações</div>
                    </div>
                    
                    <div class="mb-5">
                        <label for="default_expiration_days" class="fs-6 fw-semibold mb-2">📅 Dias de Expiração</label>
                        <input type="number" class="form-control" 
                            id="default_expiration_days" name="default_expiration_days" 
                            value="' . htmlspecialchars($currentConfig['default_expiration_days']) . '"
                            min="1" max="365" required>
                        <div class="form-text">Número de dias até a expiração (1-365)</div>
                    </div>
                    
                    <div class="notice d-flex bg-light-info rounded border-info border border-dashed p-6 mt-6">
                        <i class="ki-outline ki-information-5 fs-2x text-info me-4"></i>
                        <div class="d-flex flex-stack flex-grow-1">
                            <div class="fw-semibold">
                                <h4 class="text-gray-900 fw-bold">ℹ️ Sobre estas configurações</h4>
                                <div class="fs-6 text-gray-700">
                                    <ul>
                                        <li><strong>Status Padrão</strong>: Define se novas ativações começam como ativas ou inativas</li>
                                        <li><strong>Dias de Expiração</strong>: Define por quantos dias uma ativação será válida</li>
                                        <li>Estas configurações são salvas no arquivo <code>api/config.json</code></li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="d-flex justify-content-between mt-8">
        <button type="submit" class="btn btn-primary btn-lg">
            <i class="ki-outline ki-check fs-2 me-2"></i> Salvar Configurações
        </button>
        
        <a href="javascript:history.back()" class="btn btn-secondary btn-lg">
            <i class="ki-outline ki-arrow-left fs-2 me-2"></i> Voltar
        </a>
    </div>
</form>

<div class="separator separator-dashed my-10"></div>

<div class="row">
    <div class="col-xl-12">
        <h4 class="mb-4">📋 Configurações Atuais</h4>
        <div class="card card-flush">
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-row-bordered gy-5">
                        <thead>
                            <tr class="fw-bold fs-6 text-gray-800">
                                <th>Configuração</th>
                                <th>Valor Atual</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td><strong>Status Padrão</strong></td>
                                <td><span class="badge badge-light-' . ($currentConfig['default_status'] === 'active' ? 'success' : 'danger') . '">' . 
                                    strtoupper($currentConfig['default_status']) . '</span></td>
                            </tr>
                            <tr>
                                <td><strong>Dias de Expiração</strong></td>
                                <td>' . $currentConfig['default_expiration_days'] . ' dias</td>
                            </tr>
                            <tr>
                                <td><strong>Arquivo de Configuração</strong></td>
                                <td><code>api/config.json</code></td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>';

include 'includes/layout.php';
?>